using System;
using System.Xml;
using System.Collections;
using System.Collections.Specialized;

namespace gov.va.med.vbecs.DAL.VistALink.OpenLibrary
{
	/// <summary>
	/// Collection of RPC parameters utilized in the <see cref="RpcRequest"/> class
	/// to hold remote procedure parameters.
	/// </summary>
	public class RpcParameterCollection : NameObjectCollectionBase
	{
		// Constants used in XML message serialization/deserialization
		private const string XMLCONSTS_PARAMS_NODE_NAME = "Params";

		/// <summary>
		/// Default constructor. No specific initialization is done here. 
		/// </summary>
		public RpcParameterCollection() 
			: base( GlobalConfig.MaxRpcParametersNumber ){}

		/// <summary>
		/// XML deserialization factory method creating an instance of class from supplied parent XML element.
		/// Parent XML element must contain one and only one instance of serialized class among its children. 
		/// </summary>
		/// <param name="parentElement">
		///		Parent XML element to parse. It must contain one and 
		///		only one instance of serialized class among its children. 
		///	</param>
		/// <param name="parameterFactory">Factory object parsing source XML and creating appropriate RPC parameter objects.</param>
		/// <returns>Instance of class deserialized from a given parent <see cref="XmlElement"/>.</returns>
		public static RpcParameterCollection Parse( XmlElement parentElement, RpcParameterXmlFactory parameterFactory ) 
		{
			if( parentElement == null )
				throw( new ArgumentNullException( "parentElement" ) );

			if( parameterFactory == null )
				throw( new ArgumentNullException( "parameterFactory" ) );

			XmlElement _paramsElement = XmlUtility.ParseGetRequiredElementByUniqueTagName( parentElement, XMLCONSTS_PARAMS_NODE_NAME );

			RpcParameterCollection _result = new RpcParameterCollection();

			foreach( XmlNode _paramNode in XmlUtility.ParseGetChildElements( _paramsElement ) )
			{
				RpcParameter _paramToAdd = parameterFactory.ParseCreateRpcParameter( _paramNode );

				if( _paramToAdd == null )
					throw( new XmlParseException( SR.Exceptions.UnknownRpcParameterXmlFound( typeof(RpcParameterCollection).Name, _paramNode.InnerXml ) ) );
	
				try
				{
					_result.Add( _paramToAdd );
				}
				catch( ArgumentException xcp ) // there may be number of reasons to fail
				{
					throw( new XmlParseException( 
						SR.Exceptions.XmlDeserializationFailedDueToInnerException( typeof(RpcParameterCollection).Name, _paramNode.Name ), xcp ) );
				}
			}

			return _result;
		}

		/// <summary>
		/// Adds parameter to the collection. 
		/// Maximum number of RPC parameters is specified in RPC protocol. 
		/// It may be overriden in configuration file. 
		/// If number of parameters has reached its limit, 
		/// <see cref="InvalidOperationException"/> exception will be thrown.
		/// </summary>
		/// <param name="paramToAdd">RPC parameter to add</param>
		/// <returns>Reference to the added parameter</returns>
		public RpcParameter Add( RpcParameter paramToAdd )
		{
			if( paramToAdd == null ) 
				throw( new ArgumentNullException( "paramToAdd" ) );

			if( this.Count == GlobalConfig.MaxRpcParametersNumber )
				throw( new InvalidOperationException( SR.Exceptions.MaxNumberOfRpcParametersReached( GlobalConfig.MaxRpcParametersNumber ) ) );

			if( this.BaseGet( paramToAdd.Position.ToString() ) != null )
				throw( new ArgumentException( SR.Exceptions.RpcParameterCollectionDuplicateParameterPosition( paramToAdd.Position ), "paramToAdd" ) );

			BaseAdd( paramToAdd.Position.ToString(), paramToAdd );
			return paramToAdd;
		}
		
		/// <summary>
		/// XML serialization method writing out XML representation of 
		/// RPC parameters collection to supplied <see cref="XmlWriter"/>.
		/// </summary>
		/// <param name="writer">XmlWriter to use during deserialization.</param>
		public void WriteCollectionToXml( XmlWriter writer )
		{
			if( writer == null )
				throw( new ArgumentNullException( "writer" ) );

			writer.WriteStartElement( XMLCONSTS_PARAMS_NODE_NAME );

			for( int i = 0; i < this.Count; i++ )
				((RpcParameter)BaseGet( i )).WriteParameterToXml( writer );

			writer.WriteEndElement();
		}

		/// <summary>
		/// Gets RPC parameter by its position in RPC signature.
		/// </summary>
		/// <param name="parameterPosition">Integer parameter's position in RPC signature.</param>
		/// <returns>RPC parameter if parameter with such position is available in collection. Null otherwise.</returns>
		public RpcParameter GetParameterByPosition( int parameterPosition )
		{
			return GetParameterByPosition( new ParameterPosition( parameterPosition ) );
		}

		/// <summary>
		/// Gets RPC parameter by its position in RPC signature.
		/// </summary>
		/// <param name="parameterPosition">Parameter's position in RPC signature.</param>
		/// <returns>RPC parameter if parameter with such position is available in collection. Null otherwise.</returns>
		protected RpcParameter GetParameterByPosition( ParameterPosition parameterPosition )
		{
			if( parameterPosition == null )
				throw( new ArgumentNullException( "parameterPosition" ) );

			return (RpcParameter)BaseGet( parameterPosition.ToString() );			
		}

		/// <summary>
		/// Retrieves RPC parameter by its index in collection.
		/// </summary>
		public RpcParameter this[int index]
		{
			get
			{
				try
				{
					return (RpcParameter)BaseGet( index );
				}
				catch( ArgumentOutOfRangeException xcp )
				{
					throw( new IndexOutOfRangeException( xcp.Message ) );
				}
			}
		}

		/// <summary>
		/// Returns a signature of .NET method bound to parameters collection
		/// as an array of .NET types for method parameters.
		/// </summary>
		/// <returns>Array of .NET types representing signature of method bound to collection.</returns>
		public Type[] GetBoundMethodSignatureAsParametersTypesArray()
		{
			Type[] _result = new Type[ this.Count ];						

			for( int _paramPosition = 1; _paramPosition <= this.Count; _paramPosition++ )
				_result[ _paramPosition - 1 ] = RequireGetParamByPosition( _paramPosition ).BaseType;

			return _result; 
		}

		/// <summary>
		/// Retrieves parameter values in order defined by parameter's positions. 
		/// </summary>
		/// <returns>Array containing parameter's values.</returns>
		public object[] GetParametersValues()
		{
			object[] _result = new object[ this.Count ];

			for( int _paramPosition = 1; _paramPosition <= this.Count; _paramPosition++ )
				_result[ _paramPosition - 1 ] = RequireGetParamByPosition( _paramPosition ).BaseValue;

			return _result; 
		}	

		/// <summary>
		/// Gets parameter by position ensuring that it's always available 
		/// by throwing an exception if it's not found.
		/// </summary>
		/// <param name="paramPosition">Position of parameter to get.</param>
		/// <returns>Parameter found by position.</returns>
		private RpcParameter RequireGetParamByPosition( int paramPosition )
		{
			RpcParameter _param = GetParameterByPosition( paramPosition );

			if( _param == null )
				throw( new VistALinkException( SR.Exceptions.RpcParametersCollectionInconsistencyFound( paramPosition ) ) );

			return _param;
		}
	}
}
